<?php
/**
 * @package     n3t PhocaCart Category module
 *
 * @author      Pavel Poles - n3t.cz
 * @copyright   (C) 2023 - Pavel Poles - n3t.cz
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Joomla\Module\n3tPhocacartCategory\Site\Helper;

\defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;

/**
 * Helper class for n3t PhocaCart Category module
 *
 * @since  4.0.0
 */
class N3TPhocacartCategoryHelper implements DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	private static ?array $categories = null;
	private static ?array $rootCategories = [];

	private Registry $params;

	private CMSApplicationInterface $app;

	private function loadCategories(): array
	{
		if (self::$categories === null) {
			$db = $this->getDatabase();
			$user = \PhocacartUser::getUser();
			$lang = $this->app->getLanguage();

			$query = $db->getQuery(true)
				->select('c.*, null as children, 0 as current, 0 as active')
				->from('#__phocacart_categories as c')
				->join('LEFT', '#__phocacart_item_groups AS gc', 'c.id = gc.item_id AND gc.type = 2')
				->where('c.published = 1')
				->where('c.access IN (' . implode(',', $user->getAuthorisedViewLevels()) . ')')
				->where('(gc.group_id IN (' . implode(',', \PhocacartGroup::getGroupsById($user->id, 1, true)) . ') OR gc.group_id IS NULL)')
				->where('c.language IN (' . $db->quote($lang->getTag()) . ', ' . $db->quote('*') . ')')
				->where('c.type IN (0, 1)')
				->order('c.ordering');

			$db->setQuery($query);
			$categories = $db->loadObjectList('id');
			$rootCategories = [];
			$currentCategoryId = \PhocacartCategory::getActiveCategoryId();

			array_walk($categories, function($category) use ($categories, &$rootCategories, $currentCategoryId) {
				if ($category->id === $currentCategoryId) {
					$category->current = 1;
					$parent = $category;
					while ($parent) {
						$parent->active = 1;
						$parent->expanded = 1;
						$parent = $categories[$parent->parent_id] ?? null;
					}
				}

				if ($category->parent_id) {
					if ($categories[$category->parent_id]->children === null)
						$categories[$category->parent_id]->children = [];
					$categories[$category->parent_id]->children[] = $category;
				} else {
					$rootCategories[] = $category;
				}
			});

			self::$categories = $categories;
			self::$rootCategories = $rootCategories;
		}

		return self::$categories;
	}

	public function isExpanded(object $category): bool
	{
		if ($this->params->get('expand_subcategories')) {
			return true;
		}

		if ($category->active) {
			return true;
		}

		$expandedCategories = $this->params->get('expanded_categories', []);
		if (in_array($category->id, $expandedCategories)) {
			return true;
		}

		return false;
	}

	public function getCategories(): array
	{
		if (!ComponentHelper::isEnabled('com_phocacart')) {
			$this->app->enqueueMessage(Text::_('Phoca Cart Error'), Text::_('Phoca Cart is not installed on your system'), 'error');
			return [];
		}

		\JLoader::registerPrefix('Phocacart', JPATH_ADMINISTRATOR . '/components/com_phocacart/libraries/phocacart');

		$this->app->getLanguage()->load('com_phocacart');

		$this->loadCategories();

		if ($this->params->get('root_category')) {
			$category = self::$categories[$this->params->get('root_category')] ?? null;
			if ($category) {
				return $category->children;
			} else {
				return [];
			}
		}

		return self::$rootCategories;
	}

	public function categoryLink(object $category): ?string
	{
		return Route::_(\PhocacartRoute::getCategoryRoute($category->id, $category->alias));
	}

	/**
	 * @param   CMSApplicationInterface  $app
	 *
	 * @return N3TPhocacartCategoryHelper
	 * @since 4.0.0
	 */
	public function setApplication(CMSApplicationInterface $app): N3TPhocacartCategoryHelper
	{
		$this->app = $app;

		return $this;
	}

	/**
	 * @param   Registry  $params
	 *
	 * @return N3TPhocacartCategoryHelper
	 * @since 4.0.0
	 */
	public function setParams(Registry $params): N3TPhocacartCategoryHelper
	{
		$this->params = $params;

		return $this;
}
}
