<?php
/**
 * @package     n3t PhocaCart Category module
 *
 * @author      Pavel Poles - n3t.cz
 * @copyright   (C) 2023 - Pavel Poles - n3t.cz
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Joomla\Module\n3tPhocacartCategory\Site\Helper;

\defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;

/**
 * Helper class for n3t PhocaCart Category module
 *
 * @since  4.0.0
 */
class N3TPhocacartCategoryHelper implements DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	private static ?array $categories = null;
	private static ?array $rootCategories = [];

	private Registry $params;

	private CMSApplicationInterface $app;

	/**
	 * Checks if PhocaCart is installed and enabled
	 *
	 * @return bool
	 *
	 * @since 4.0.0
	 */
	public function isPhocaCart(): bool
	{
		return ComponentHelper::isEnabled('com_phocacart');
	}

	/**
	 * Loads PhocaCart classes
	 *
	 * @return bool
	 *
	 * @since 4.0.0
	 */
	public function loadPhocaCart(): bool
	{
		if (!$this->isPhocaCart()) {
			$this->app->enqueueMessage(Text::_('Phoca Cart is not installed on your system'), 'error');
			return false;
		}

		\JLoader::registerPrefix('Phocacart', JPATH_ADMINISTRATOR . '/components/com_phocacart/libraries/phocacart');
		return true;
	}

	/**
	 * Loads categories cache
	 *
	 * @return array
	 *
	 * @since 4.0.0
	 */
	private function loadCategories(): array
	{
		if (self::$categories === null) {
			$db = $this->getDatabase();
			$user = \PhocacartUser::getUser();
			$lang = $this->app->getLanguage();

			$query = $db->getQuery(true)
				->select('c.*, null AS children, 0 AS current, 0 AS active, 1 AS level')
				->from('#__phocacart_categories as c')
				->join('LEFT', '#__phocacart_item_groups AS gc', 'c.id = gc.item_id AND gc.type = 2')
				->where('c.published = 1')
				->where('c.access IN (' . implode(',', $user->getAuthorisedViewLevels()) . ')')
				->where('(gc.group_id IN (' . implode(',', \PhocacartGroup::getGroupsById($user->id, 1, true)) . ') OR gc.group_id IS NULL)')
				->where('c.language IN (' . $db->quote($lang->getTag()) . ', ' . $db->quote('*') . ')')
				->where('c.type IN (0, 1)')
				->order('c.parent_id, c.ordering');

			$db->setQuery($query);
			$categories = $db->loadObjectList('id');
			$rootCategories = [];
			$currentCategoryId = \PhocacartCategory::getActiveCategoryId();

			array_walk($categories, function($category) use ($categories, &$rootCategories, $currentCategoryId) {
				if ($category->id === $currentCategoryId) {
					$category->current = 1;
					$parent = $category;
					while ($parent) {
						$parent->active = 1;
						$parent->expanded = 1;
						$parent = $categories[$parent->parent_id] ?? null;
					}
				}

				if ($category->parent_id) {
					$category->level = $categories[$category->parent_id]->level + 1;
					if ($categories[$category->parent_id]->children === null)
						$categories[$category->parent_id]->children = [];
					$categories[$category->parent_id]->children[] = $category;
				} else {
					$rootCategories[] = $category;
				}
			});

			self::$categories = $categories;
			self::$rootCategories = $rootCategories;
		}

		return self::$categories;
	}

	/**
	 * Checks if given node should be expanded
	 *
	 * @param   object  $category
	 *
	 * @return bool
	 *
	 * @since 4.0.0
	 */
	public function isExpanded(object $category): bool
	{
		if ($this->params->get('expand_subcategories')) {
			return true;
		}

		if ($category->active) {
			return true;
		}

		$expandedCategories = $this->params->get('expanded_categories', []);
		if (in_array($category->id, $expandedCategories)) {
			return true;
		}

		return false;
	}

	/**
	 * Returns categories based on module settings
	 *
	 * @return array
	 *
	 * @since 4.0.0
	 */
	public function getCategories(): array
	{
		if (!$this->loadPhocaCart()) {
			return [];
		}

		$this->app->getLanguage()->load('com_phocacart');

		$this->loadCategories();

		if ($this->params->get('root_category')) {
			$category = self::$categories[$this->params->get('root_category')] ?? null;
			if ($category) {
				$categories = $category->children;
			} else {
				$categories = [];
			}
		} else {
			$categories = self::$rootCategories;
		}

		if ($this->params->get('start_level',  1) > 1) {
			$currentCategoryId = \PhocacartCategory::getActiveCategoryId();
			if ($currentCategoryId) {
				$currentCategory = self::$categories[$currentCategoryId] ?? null;
				while ($currentCategory && $currentCategory->level >= $this->params->get('start_level',  1)) {
					$currentCategory = self::$categories[$currentCategory->parent_id] ?? null;
				}
				if ($currentCategory) {
					$categories = $currentCategory->children;
				} else {
					$categories = [];
				}
			} else {
				$categories = [];
			}
		}

		return $categories;
	}

	/**
	 * Returns link to category
	 *
	 * @param   object  $category
	 *
	 * @return string|null
	 *
	 * @since 4.0.0
	 */
	public function categoryLink(object $category): ?string
	{
		return Route::_(\PhocacartRoute::getCategoryRoute($category->id, $category->alias));
	}

	/**
	 * Fluent application setter
	 *
	 * @param   CMSApplicationInterface  $app
	 *
	 * @return N3TPhocacartCategoryHelper
	 * @since 4.0.0
	 */
	public function setApplication(CMSApplicationInterface $app): N3TPhocacartCategoryHelper
	{
		$this->app = $app;

		return $this;
	}

	/**
	 * Fluent params setter
	 *
	 * @param   Registry  $params
	 *
	 * @return N3TPhocacartCategoryHelper
	 * @since 4.0.0
	 */
	public function setParams(Registry $params): N3TPhocacartCategoryHelper
	{
		$this->params = $params;

		return $this;
	}

	/**
	 * Truncates html text to plain text with given length
	 *
	 * @param   string|null  $html
	 * @param   int          $maxLength
	 *
	 * @return string
	 *
	 * @since 4.0.1
	 */
	public function truncate(?string $html, int $maxLength = 0, string $extension = '…'): string
	{
		$html = $html ?: '';
		if ($maxLength <= 0) {
			return '';
		}

		$string = HTMLHelper::_('string.truncate', $html, $maxLength, $noSplit = true, $allowHtml = false);
		return preg_replace('~\.\.\.$~', $extension, $string);
	}

}
