/**
 * @package     n3t PhocaCart Search module
 *
 * @author      Pavel Poles - n3t.cz
 * @copyright   (C) 2023-2024 - Pavel Poles - n3t.cz
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 **/
(function () {
  'use strict';

  if (!Joomla || !Joomla.Text) {
    throw new Error('core.js was not properly initialised');
  }

  function runAutocomplete(autocompleter, input, url) {
    autocompleter.classList.add('show', 'loading');
    autocompleter.classList.remove('error');
    url.searchParams.set('search', input.value);

    Joomla.request({
      url: url.toString(),
      perform: true,
      onSuccess: function(response) {
        autocompleter.innerHTML = response
      },
      onError: function() {
        autocompleter.innerHTML = '<div>' + Joomla.Text._('MOD_N3T_PHOCACART_SEARCH_JS_RESPONSE_ERROR') + '</div>';
        autocompleter.classList.add('error');
      },
      onComplete: function() {
        autocompleter.classList.remove('loading');
      }
    });
  }

  function hideAutocompleter(autocompleter) {
    autocompleter.classList.remove('show', 'error', 'loading');
    autocompleter.innerHTML = '';
  }

  Array.from(document.querySelectorAll('[data-n3t-ajax-search]')).forEach(function(form) {
    var input = form.querySelector('input[type=search]'),
      searchUrl = new URL(form.dataset.n3tAjaxSearch),
      autocompleter = document.createElement('div'),
      timeout = form.dataset.n3tAjaxSearchTimeout ? form.dataset.n3tAjaxSearchTimeout : 500,
      minLength = form.dataset.n3tAjaxSearchLength ? form.dataset.n3tAjaxSearchLength : 3;

    var timer = 0;

    autocompleter.classList.add('n3t_phocacart_search__autocompleter');
    form.appendChild(autocompleter);

    input.addEventListener('input', function(e) {
      if (timer)
        window.clearTimeout(timer);
      if (input.value.length >= minLength) {
        timer = window.setTimeout(function () {
          runAutocomplete(autocompleter, input, searchUrl)
        }, timeout);
      } else {
        hideAutocompleter(autocompleter);
      }
    });

    document.body.addEventListener('click', function(e) {
      if (e.target.closest('[data-n3t-ajax-search]') !== form)
        hideAutocompleter(autocompleter);
    });

    window.addEventListener('keydown', function(e) {
      if (!['ArrowUp', 'ArrowDown', 'Escape'].includes(e.code))
        return;

      if (!autocompleter.classList.contains('show'))
        return;

      if (e.code === 'Escape') {
        hideAutocompleter(autocompleter);
        e.preventDefault();
        return;
      }

      var currentEl = document.activeElement;
      var targetEl = null;

      if (currentEl.closest('form') !== form)
        return;

      var targetList = Array.from(autocompleter.querySelectorAll('.n3t_phocacart_search__products a, .n3t_phocacart_search__total a'));

      if (!targetList.length)
        return;

      if (currentEl.closest('.n3t_phocacart_search__autocompleter') !== autocompleter) {
        if (e.code === 'ArrowDown')
          targetEl = targetList[0];
      } else {
        var index = targetList.indexOf(currentEl);

        if(e.code === 'ArrowDown' && index < targetList.length - 1) {
          targetEl = targetList[index + 1];
        } else if (e.code === 'ArrowUp' && index > 0) {
          targetEl = targetList[index - 1];
        }
      }

      if (targetEl) {
        targetEl.focus();
        e.preventDefault();
      }
    });
  });
})();
