<?php
/**
 * @package     n3t PhocaCart Search module
 *
 * @author      Pavel Poles - n3t.cz
 * @copyright   (C) 2023-2024 - Pavel Poles - n3t.cz
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Joomla\Module\n3tPhocaCartSearch\Site\Helper;

\defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Event\Plugin\System\Schemaorg\BeforeCompileHeadEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\ParameterType;
use Joomla\Event\DispatcherAwareInterface;
use Joomla\Event\DispatcherAwareTrait;
use Joomla\Event\Priority;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;

/**
 * Helper class for n3t PhocaCart Search module
 *
 * @since  4.0.0
 */
class N3TPhocacartSearchHelper implements DatabaseAwareInterface, SubscriberInterface, DispatcherAwareInterface
{
	use DatabaseAwareTrait, DispatcherAwareTrait;

	private Registry $params;

	private Registry $phocaCartParams;

	private bool $isI18n = false;

	private CMSApplicationInterface $app;

	private ?\stdClass $module = null;

	/**
	 * Search for correct module instance
	 *
	 * @return object
	 *
	 * @throws \Exception
	 * @since 4.0.0
	 */
	private function getModule(): object
	{
		if ($this->module === null) {

			$app      = Factory::getApplication();
			$moduleId = $app->getInput()->get('id', 0, 'UINT');
			if ($moduleId) {
				$this->module = ModuleHelper::getModuleById((string) $moduleId);
				if ($this->module->module !== 'mod_n3t_phocacart_search')
					$this->module = null;
			}

			if (!$this->module) {
				$this->module = ModuleHelper::getModule('mod_n3t_phocacart_search');
			}
		}

		return $this->module;
	}

	/**
	 * Checks, if PhocaCart is installed and enabled
	 *
	 * @return bool
	 *
	 * @since 4.0.0
	 */
	public function isPhocaCart(): bool
	{
		return ComponentHelper::isEnabled('com_phocacart');
	}

	/**
	 * Loads PhocaCart classes
	 *
	 * @return bool
	 *
	 * @since 4.0.0
	 */
	public function loadPhocaCart(): bool
	{
		if (!$this->isPhocaCart()) {
			$this->app->enqueueMessage(Text::_('Phoca Cart is not installed on your system'), 'error');
			return false;
		}

		if (file_exists(JPATH_ADMINISTRATOR . '/components/com_phocacart/libraries/bootstrap.php')) {
			require_once JPATH_ADMINISTRATOR . '/components/com_phocacart/libraries/bootstrap.php';
		} else {
			\JLoader::registerPrefix('Phocacart', JPATH_ADMINISTRATOR . '/components/com_phocacart/libraries/phocacart');
		}

		return true;
	}

	/**
	 * Dispatch module content
	 *
	 * @param   \stdClass  $module
	 * @param   array      $displayData
	 * @param   string     $layout
	 *
	 * @return string
	 *
	 * @since 4.0.0
	 */
	private function dispatch(\stdClass $module, array $displayData, string $layout): string
	{
		$loader = static function (\stdClass $module, array $displayData, string $layout) {
			if (!\array_key_exists('displayData', $displayData)) {
				extract($displayData);
				unset($displayData);
			} else {
				extract($displayData);
			}

			require ModuleHelper::getLayoutPath($module->module, $layout);
		};

		ob_start();
		$loader($module, $displayData, $layout);
		return ob_get_clean();
	}

	/**
	 * Gets SQL ordering based on numeric settings value
	 *
	 * @param   int  $ordering
	 *
	 * @return string
	 *
	 * @since 4.0.0
	 */
	private function getOrdering(int $ordering): string
	{
		switch ($ordering) {
			case 1:
			default: return 'pc.ordering ASC, p.ordering ASC, p.id ASC';
			case 2: return 'pc.ordering DESC, p.ordering DESC, p.id ASC';
			case 3: return 'p.title ASC, p.ordering ASC, p.id ASC';
			case 4: return 'p.title DESC, p.ordering ASC, p.id ASC';
			case 5: return 'p.price ASC, p.ordering ASC, p.id ASC';
			case 6: return 'p.price DESC, p.ordering ASC, p.id ASC';
			case 7: return 'p.date ASC, p.ordering ASC, p.id ASC';
			case 8: return 'p.date DESC, p.ordering ASC, p.id ASC';
			case 9: return 'rating ASC, p.ordering ASC, p.id ASC';
			case 10: return 'rating DESC, p.ordering ASC, p.id ASC';
			// Not used
			//case 11: return 'p.id ASC';
			//case 12: return 'p.id DESC';
			case 13: return 'p.sales ASC, p.ordering ASC, p.id ASC';
			case 14: return 'p.sales DESC, p.ordering ASC, p.id ASC';
			case 15: return 'p.hits ASC, p.ordering ASC, p.id ASC';
			case 16: return 'p.hits DESC, p.ordering ASC, p.id ASC';
			// Not used
			//case 17: return 'ph.hits ASC, p.ordering ASC, p.id ASC';
			//case 18: return 'ph.hits DESC, p.ordering ASC, p.id ASC';
			case 19: return 'p.sku ASC, p.ordering ASC, p.id ASC';
			case 20: return 'p.sku DESC, p.ordering ASC, p.id ASC';
			case 21: return 'p.date_update ASC, p.ordering ASC, p.id ASC';
			case 22: return 'p.date_update DESC, p.ordering ASC, p.id ASC';
			case 23: return 'p.stock ASC, p.ordering ASC, p.id ASC';
			case 24: return 'p.stock DESC, p.ordering ASC, p.id ASC';
			case 25: return 'p.featured ASC, p.ordering ASC, p.id ASC';
			case 26: return 'p.featured DESC, p.ordering ASC, p.id ASC';
			// Not used
			//case 99: return 'RAND()';
		}
	}

	/**
	 * Prepares array of SQL keywords, based on entered text
	 *
	 * @param   string|null  $search
	 *
	 * @return array
	 *
	 * @since 4.0.1
	 */
	private function prepareKeywords(?string $search): array
	{
		if (!$search) {
			return [];
		}

		switch ($this->phocaCartParams->get('search_matching_option', 'any')) {
			case 'exact':
				$keywords = [$search];
				break;
			case 'all':
			case 'any':
				$keywords = explode(' ', $search);
				break;
		}

		array_walk($keywords, function(string &$keyword) {
			$keyword = trim($keyword);
		});

		$keywords = array_filter($keywords, function(string $keyword) {
			return !!$keyword;
		});

		if (!$keywords) {
			return [];
		}

		$db = $this->getDatabase();
		array_walk($keywords, function(string &$keyword) use ($db) {
			$keyword = $db->quote('%' . $db->escape($keyword) . '%');
		});

		return $keywords;
	}

	/**
	 * Search products based on given keywords
	 *
	 * @param   array  $keywords
	 * @param   int    $totalCount
	 *
	 * @return array
	 *
	 * @since 4.0.0
	 */
	private function searchProducts(array $keywords, int &$totalCount = 0): array
	{
		if (!$keywords) {
			return [];
		}

		if (!$this->params->get('search_products', 1)) {
			return [];
		}

		$user = \PhocacartUser::getUser();
		$ordering = $this->phocaCartParams->get('item_ordering', 1);
		$limit = $this->params->get('limit_results', 10);
		$lang = $this->app->getLanguage();

		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->select(
				'DISTINCT SQL_CALC_FOUND_ROWS pc.product_id'
			)
			->from('#__phocacart_product_categories AS pc')
			->join('LEFT', '#__phocacart_products AS p', 'p.id = pc.product_id')
			->join('LEFT', '#__phocacart_categories AS pcc', 'pcc.id = pc.category_id')
			->where([
				'p.published = 1',
				'pcc.published = 1',
			])
			->order($this->getOrdering($ordering))
			->setLimit($limit);

		if ($this->isI18n) {
			$query
				->join('LEFT', '#__phocacart_products_i18n AS i18n', 'i18n.id = p.id AND i18n.language = ' . $db->quote($lang->getTag()));
		}

		if (in_array($ordering, [9, 10])) {
			$query
				->select('AVG(r.rating) AS rating')
				->join('LEFT', '#__phocacart_reviews AS r', 'r.product_id = p.id AND r.id > 0');
		}

		if (!$this->phocaCartParams->get('sql_products_skip_category_type')) {
			$query->whereIn('pcc.type', [0, 1]);
		}

		if ($this->app->getLanguageFilter()) {
			$query
				->whereIn('p.language', [$lang->getTag(), '*'], ParameterType::STRING)
				->whereIn('pcc.language', [$lang->getTag(), '*'], ParameterType::STRING);
		}

		if (!$this->phocaCartParams->get('sql_products_skip_access')) {
			$query
				->whereIn('p.access', $user->getAuthorisedViewLevels())
				->whereIn('pcc.access', $user->getAuthorisedViewLevels());
		}

		if (in_array($ordering, [9, 10])) {
			$query
				->select('AVG(r.rating) AS rating')
				->join('LEFT', '#__phocacart_reviews AS r', 'r.product_id = p.id AND r.id > 0');
		}

		if (!$this->phocaCartParams->get('sql_products_skip_group')) {
			$userGroups = implode (',', \PhocacartGroup::getGroupsById($user->id, 1, 1));

			$query
				->join('LEFT', '#__phocacart_item_groups AS gp', 'gp.item_id = p.id AND gp.type = 3')
				->where('(gp.group_id IN (' . $userGroups . ') OR gp.group_id IS NULL)')

				->join('LEFT', '#__phocacart_item_groups AS gc', 'gc.item_id = pc.category_id AND gc.type = 2')
				->where('(gc.group_id IN (' . $userGroups . ') OR gc.group_id IS NULL)');
		}

		if ($this->phocaCartParams->get( 'hide_products_out_of_stock')) {
			$query->where('p.stock > 0');
		}

		if (!in_array($this->phocaCartParams->get('sql_search_skip_id', 1), [1, 2])) {
			$query->join('LEFT', '#__phocacart_product_stock AS ps', 'ps.product_id = p.id');
		}

		$subQuery = [];
		foreach ($keywords as $keyword) {
			if ($this->isI18n) {
				$conditions = [
					'coalesce(i18n.title, p.title) LIKE ' . $keyword,
					'coalesce(i18n.alias, p.alias) LIKE ' . $keyword,
					'coalesce(i18n.title_long, p.title_long) LIKE ' . $keyword,
					'coalesce(i18n.metatitle, p.metatitle) LIKE ' . $keyword,
					'coalesce(i18n.metakey, p.metakey) LIKE ' . $keyword,
					'coalesce(i18n.metadesc, p.metadesc) LIKE ' . $keyword,
					'coalesce(i18n.description, p.description) LIKE ' . $keyword,
					'p.sku LIKE ' . $keyword,
					'p.ean LIKE ' . $keyword,
				];

				if ($this->phocaCartParams->get('search_deep')) {
					$conditions[] = 'coalesce(i18n.description_long, p.description_long) LIKE ' . $keyword;
					$conditions[] = 'coalesce(i18n.features, p.features) LIKE ' . $keyword;
				}

				if (!in_array($this->phocaCartParams->get('sql_search_skip_id', 1), [1, 2])) {
					$conditions[] = 'ps.sku LIKE ' . $keyword;
					$conditions[] = 'ps.ean LIKE ' . $keyword;
				}
			} else {
				$conditions = [
					'p.title LIKE ' . $keyword,
					'p.alias LIKE ' . $keyword,
					'p.metatitle LIKE ' . $keyword,
					'p.metakey LIKE ' . $keyword,
					'p.metadesc LIKE ' . $keyword,
					'p.description LIKE ' . $keyword,
					'p.sku LIKE ' . $keyword,
					'p.ean LIKE ' . $keyword,
				];

				if (version_compare(\PhocacartUtils::getPhocaVersion(), '4.0.6', '>=')) {
					$conditions[] = 'p.title_long LIKE ' . $keyword;
				}

				if ($this->phocaCartParams->get('search_deep')) {
					$conditions[] = 'p.description_long LIKE ' . $keyword;
					$conditions[] = 'p.features LIKE ' . $keyword;
				}

				if (!in_array($this->phocaCartParams->get('sql_search_skip_id', 1), [1, 2])) {
					$conditions[] = 'ps.sku LIKE ' . $keyword;
					$conditions[] = 'ps.ean LIKE ' . $keyword;
				}
			}

			$subQuery[] = '(' . implode(' OR ', $conditions) . ')';
		}

		$query->extendWhere('AND', $subQuery, $this->phocaCartParams->get('search_matching_option', 'any') == 'any' ? 'OR' : 'AND');

		$db->setQuery($query);
		$productIds = $db->loadColumn();

		$db->setQuery('SELECT FOUND_ROWS()');
		$totalCount = $db->loadResult();

		if (!$productIds) {
			return [];
		}

		$query = $db->getQuery(true)
			->select('p.*, p.catid AS pref_cat_id, pcc.id AS cat_id')
			->from('#__phocacart_products AS p')
			->join('LEFT', '#__phocacart_product_categories AS pc', 'pc.product_id = p.id')
			->join('LEFT', '#__phocacart_categories AS pcc', 'pcc.id = pc.category_id')
			->order($this->getOrdering($ordering))
			->whereIn('p.id', $productIds)
			->group('p.id');

		if ($this->isI18n) {
			$query
				->select('coalesce(i18n.alias, p.alias) AS alias, coalesce(i18n.title, p.title) AS title, coalesce(i18n.title_long, p.title_long) AS title_long')
				->select('coalesce(i18n_pcc.alias, pcc.alias) AS cat_alias, coalesce(i18n_pcc.title, c.title) AS cat_title')
				->join('LEFT', '#__phocacart_products_i18n AS i18n', 'i18n.id = p.id AND i18n.language = ' . $db->quote($lang->getTag()))
				->join('LEFT', '#__phocacart_categories_i18n AS i18n_pcc', 'i18n_pcc.id = pcc.id AND i18n_pcc.language = ' . $db->quote($lang->getTag()));
		} else {
			$query
				->select('pcc.alias AS cat_alias, pcc.title AS cat_title');
		}

		if ($this->params->get('show_categories', 1)) {
			$query
				->join('LEFT', '#__phocacart_categories AS c', 'c.id = p.catid');
			if ($this->isI18n) {
				$query
					->select('coalesce(i18n_c.alias, c.alias) AS pref_cat_alias, coalesce(i18n_c.title, c.title) AS pref_cat_title')
					->join('LEFT', '#__phocacart_categories_i18n AS i18n_c', 'i18n_c.id = c.id AND i18n_c.language = ' . $db->quote($lang->getTag()));
			} else {
				$query
					->select('c.alias AS pref_cat_alias, c.title AS pref_cat_title');
			}
		}

		if ($this->params->get('show_manufacturers', 1)) {
			$query
				->join('LEFT', '#__phocacart_manufacturers AS m', 'm.id = p.manufacturer_id');
			if ($this->isI18n) {
				$query
					->select('coalesce(i18n_m.alias, m.alias) AS manufacturer_alias, coalesce(i18n_m.title, m.title) AS manufacturer_title')
					->join('LEFT', '#__phocacart_manufacturers_i18n AS i18n_m', 'i18n_m.id = m.id AND i18n_m.language = ' . $db->quote($lang->getTag()));
			} else {
				$query
					->select('m.alias AS manufacturer_alias, m.title AS manufacturer_title');
			}
		}

		if ($this->params->get('show_prices', 1)) {
			$query->select('t.id AS taxid, t.tax_rate AS taxrate, t.calculation_type AS taxcalculationtype, t.title AS taxtitle')
				->join('LEFT', '#__phocacart_taxes AS t', 't.id = p.tax_id');

			if (version_compare(\PhocacartUtils::getPhocaVersion(), '4.0.6','>=')) {
				$query->select('t.tax_hide AS taxhide');
			}
		}

		if (!$this->phocaCartParams->get('sql_products_skip_group')) {
			$userGroups = implode (',', \PhocacartGroup::getGroupsById($user->id, 1, 1));

			$query
				->join('LEFT', '#__phocacart_item_groups AS gp', 'gp.item_id = p.id AND gp.type = 3')
				->where('(gp.group_id IN (' . $userGroups . ') OR gp.group_id IS NULL)');

			if ($this->params->get('show_prices', 1)) {
				$query
					->select('MIN(ppg.price) AS group_price')
					->join('LEFT', '#__phocacart_product_price_groups AS ppg', 'ppg.product_id = p.id AND ppg.group_id = gp.group_id');
			}
		} elseif ($this->params->get('show_prices', 1)) {
			$query->select('NULL AS group_price');
		}

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Search categories based on given keywords
	 *
	 * @param   array  $keywords
	 *
	 * @return array
	 *
	 * @since 4.0.1
	 */
	private function searchCategories(array $keywords): array
	{
		if (!$keywords) {
			return [];
		}

		if (!$this->params->get('search_categories', 1)) {
			return [];
		}

		$limit = $this->params->get('limit_categories', 10);
		$lang = $this->app->getLanguage();

		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->select('c.*')
			->from('#__phocacart_categories AS c')
			->where('c.published = 1')
			->order('c.ordering ASC')
			->setLimit($limit);

		if ($this->isI18n) {
			$query
				->select('coalesce(i18n.title, c.title) AS title, coalesce(i18n.description, c.description) AS description')
				->select('coalesce(i18n.title_long, c.title_long) AS title_long, coalesce(i18n.alias, c.alias) AS alias')
				->join('LEFT', '#__phocacart_categories_i18n AS i18n', 'i18n.id = c.id AND i18n.language = ' . $db->quote($lang->getTag()));
		}

		if ($this->app->getLanguageFilter()) {
			$lang = Factory::getLanguage()->getTag();
			$query
				->whereIn('c.language', [$lang, '*'], ParameterType::STRING);
		}

		if (!$this->phocaCartParams->get('sql_products_skip_access')) {
			$user = \PhocacartUser::getUser();
			$query
				->whereIn('c.access', $user->getAuthorisedViewLevels());
		}

		$subQuery = [];
		foreach ($keywords as $keyword) {
			if ($this->isI18n) {
				$conditions = [
					'coalesce(i18n.title, c.title) LIKE ' . $keyword,
					'coalesce(i18n.alias, c.alias) LIKE ' . $keyword,
					'coalesce(i18n.metatitle, c.metatitle) LIKE ' . $keyword,
					'coalesce(i18n.metakey, c.metakey) LIKE ' . $keyword,
					'coalesce(i18n.metadesc, c.metadesc) LIKE ' . $keyword,
					'coalesce(i18n.title_long, c.title_long) LIKE ' . $keyword
				];

				if ($this->phocaCartParams->get('search_deep')) {
					$conditions[] = 'coalesce(i18n.description, c.description) LIKE ' . $keyword;
				}
			} else {
				$conditions = [
					'c.title LIKE ' . $keyword,
					'c.alias LIKE ' . $keyword,
					'c.metatitle LIKE ' . $keyword,
					'c.metakey LIKE ' . $keyword,
					'c.metadesc LIKE ' . $keyword,
				];

				if (version_compare(\PhocacartUtils::getPhocaVersion(), '4.0.6', '>=')) {
					$conditions[] = 'c.title_long LIKE ' . $keyword;
				}

				if ($this->phocaCartParams->get('search_deep')) {
					$conditions[] = 'c.description LIKE ' . $keyword;
				}
			}

			$subQuery[] = '(' . implode(' OR ', $conditions) . ')';
		}

		$query->extendWhere('AND', $subQuery, $this->phocaCartParams->get('search_matching_option', 'any') == 'any' ? 'OR' : 'AND');

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Search manufacturers based on given keywords
	 *
	 * @param   array  $keywords
	 *
	 * @return array
	 *
	 * @since 4.0.1
	 */
	private function searchManufacturers(array $keywords): array
	{
		if (!$keywords) {
			return [];
		}

		if (!$this->params->get('search_manufacturers', 1)) {
			return [];
		}

		$limit = $this->params->get('limit_manufacturers', 10);
		$lang = $this->app->getLanguage();

		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->select('m.*')
			->from('#__phocacart_manufacturers AS m')
			->where('m.published = 1')
			->order('m.ordering ASC')
			->setLimit($limit);

		if ($this->isI18n) {
			$query
				->select('coalesce(i18n.title, m.title) AS title, coalesce(i18n.description, m.description) AS description')
				->select('coalesce(i18n.title_long, m.title_long) AS title_long, coalesce(i18n.alias, m.alias) AS alias')
				->join('LEFT', '#__phocacart_manufacturers_i18n AS i18n', 'i18n.id = m.id AND i18n.language = ' . $db->quote($lang->getTag()));
		}

		if ($this->app->getLanguageFilter()) {
			$lang = Factory::getLanguage()->getTag();
			$query
				->whereIn('m.language', [$lang, '*'], ParameterType::STRING);
		}

		$subQuery = [];
		foreach ($keywords as $keyword) {
			if ($this->isI18n) {
				$conditions = [
					'coalesce(i18n.title, m.title) LIKE ' . $keyword,
					'coalesce(i18n.alias, m.alias) LIKE ' . $keyword,
					'coalesce(i18n.title_long, m.title_long) LIKE ' . $keyword,
					'coalesce(i18n.metatitle, m.metatitle) LIKE ' . $keyword,
					'coalesce(i18n.metakey, m.metakey) LIKE ' . $keyword,
					'coalesce(i18n.metadesc, m.metadesc) LIKE ' . $keyword,
				];

				if ($this->phocaCartParams->get('search_deep')) {
					$conditions[] = 'coalesce(i18n.description, m.description) LIKE ' . $keyword;
				}
			} else {
				$conditions = [
					'm.title LIKE ' . $keyword,
					'm.alias LIKE ' . $keyword,
				];

				if (version_compare(\PhocacartUtils::getPhocaVersion(), '4.0.6', '>=')) {
					$conditions[] = 'm.title_long LIKE ' . $keyword;
					$conditions[] = 'm.metatitle LIKE ' . $keyword;
					$conditions[] = 'm.metakey LIKE ' . $keyword;
					$conditions[] = 'm.metadesc LIKE ' . $keyword;
				}

				if ($this->phocaCartParams->get('search_deep')) {
					$conditions[] = 'm.description LIKE ' . $keyword;
				}
			}

			$subQuery[] = '(' . implode(' OR ', $conditions) . ')';
		}

		$query->extendWhere('AND', $subQuery, $this->phocaCartParams->get('search_matching_option', 'any') == 'any' ? 'OR' : 'AND');

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Search tags based on given keywords
	 *
	 * @param   array  $keywords
	 *
	 * @return array
	 *
	 * @since 4.0.1
	 */
	private function searchTags(array $keywords): array
	{
		if (!$keywords) {
			return [];
		}

		if (!$this->params->get('search_tags', 1)) {
			return [];
		}

		$limit = $this->params->get('limit_tags', 10);
		$lang = $this->app->getLanguage();

		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->select('t.*')
			->from('#__phocacart_tags AS t')
			->where('t.published = 1')
			->where('t.type = 0')
			->order('t.ordering ASC')
			->setLimit($limit);

		if ($this->isI18n) {
			$query
				->select('coalesce(i18n.title, t.title) AS title, coalesce(i18n.alias, t.alias) AS alias')
				->join('LEFT', '#__phocacart_tags_i18n AS i18n', 'i18n.id = t.id AND i18n.language = ' . $db->quote($lang->getTag()));
		}

		$subQuery = [];
		foreach ($keywords as $keyword) {
			if ($this->isI18n) {
				$conditions = [
					'coalesce(i18n.title, t.title) LIKE ' . $keyword,
					'coalesce(i18n.alias, t.alias) LIKE ' . $keyword,
				];
			} else {
				$conditions = [
					't.title LIKE ' . $keyword,
					't.alias LIKE ' . $keyword,
				];
			}

			$subQuery[] = '(' . implode(' OR ', $conditions) . ')';
		}

		$query->extendWhere('AND', $subQuery, $this->phocaCartParams->get('search_matching_option', 'any') == 'any' ? 'OR' : 'AND');

		$db->setQuery($query);

		return $db->loadObjectList();
	}

	/**
	 * Process AJAX request
	 *
	 * @return string
	 *
	 * @throws \Exception
	 * @since 4.0.0
	 */
	public function getAjax(): string
	{
		if (!$this->loadPhocaCart()) {
			return '';
		}

		$this->app = Factory::getApplication();
		$module = $this->getModule();
		$this->params = new Registry($module->params);
		$search = $this->app->getInput()->get('search', null, 'string');

		$lang = $this->app->getLanguage();
		$lang->load($module->module);

		$this->phocaCartParams = ComponentHelper::getParams('com_phocacart');
		$this->isI18n = !!$this->phocaCartParams->get('i18n');

		$displayData = [
			'module'          => $this->module,
			'app'             => $this->app,
			'input'           => $this->app->getInput(),
			'params'          => $this->params,
			'phocaCartParams' => $this->phocaCartParams,
			'template'        => $this->app->getTemplate()
		];

		$keywords = $this->prepareKeywords($search);

		$totalCount = 0;
		$displayData['currentSearch'] = $search;
		$displayData['products'] = $this->searchProducts($keywords, $totalCount);
		$displayData['productsCount'] = $totalCount;
		$displayData['categories'] = $this->searchCategories($keywords);;
		$displayData['manufacturers'] = $this->searchManufacturers($keywords);
		$displayData['tags'] = $this->searchTags($keywords);
		$displayData['helper'] = $this;

		return $this->dispatch($module, $displayData, $this->params->get('layout', 'default') . '_ajax');
	}

	/**
	 * Params fluent setter
	 *
	 * @param   Registry  $params
	 *
	 * @return $this
	 *
	 * @since 4.0.0
	 */
	public function setParams(Registry $params): N3TPhocacartSearchHelper
	{
		$this->params = $params;

		return $this;
	}

	/**
	 * Application fluent setter
	 *
	 * @param   CMSApplicationInterface  $app
	 *
	 * @return N3TPhocacartSearchHelper
	 *
	 * @since 4.0.0
	 */
	public function setApplication(CMSApplicationInterface $app): N3TPhocacartSearchHelper
	{
		$this->app = $app;

		return $this;
	}

	/**
	 * Gets product price array
	 *
	 * @param   object  $product
	 *
	 * @return array
	 *
	 * @since 4.0.2
	 */
	public function getProductPrice(object $product): array
	{
		$priceHelper = new \PhocacartPrice;

		if (version_compare(\PhocacartUtils::getPhocaVersion(), '4.0.6','>=')) {
			return $priceHelper->getPriceItems($product->price, $product->taxid, $product->taxrate, $product->taxcalculationtype, $product->taxtitle,
				$product->unit_amount, $product->unit_unit, 1, 1, $product->group_price, $product->taxhide);
		}

		return $priceHelper->getPriceItems($product->price, $product->taxid, $product->taxrate, $product->taxcalculationtype, $product->taxtitle,
			$product->unit_amount, $product->unit_unit, 1, 1, $product->group_price);
	}

	/**
	 * Gets product original price array
	 *
	 * @param   object  $product
	 *
	 * @return array
	 *
	 * @since 4.0.2
	 */
	public function getProductPriceOriginal(object $product): array
	{
		if (!(int)$product->price_original) {
			return [];
		}

		$priceHelper = new \PhocacartPrice;

		if (version_compare(\PhocacartUtils::getPhocaVersion(), '4.0.6','>=')) {
			return $priceHelper->getPriceItems($product->price_original, $product->taxid, $product->taxrate, $product->taxcalculationtype,
				'', 0, '', 0, 1, null, $product->taxhide);
		}

		return $priceHelper->getPriceItems($product->price_original, $product->taxid, $product->taxrate, $product->taxcalculationtype,
			'', 0, '', 0, 1, null);
	}

	/**
	 * @param   string|null  $search
	 *
	 * @return string
	 *
	 * @since 4.0.3
	 */
	public function getItemsRoute(?string $search = null): string
	{
		$link = 'index.php?option=com_phocacart&view=items';

		if ($search) {
			$link .= '&search=' . urlencode($search);
		}

		if ($this->params->get('item_id')) {
			$link .= '&Itemid=' . (int)$this->params->get('item_id');
		}

		return $link;
	}

	/**
	 * @param   int     $manufacturerId
	 * @param   string  $manufacturerAlias
	 *
	 * @return string
	 *
	 * @since 4.0.3
	 */
	public function getManufacturerRoute(int $manufacturerId, string $manufacturerAlias): string
	{
		$link = 'index.php?option=com_phocacart&view=items';
		$link .= '&' . $this->phocaCartParams->get('manufacturer_alias', 'manufacturer') . '=' . $manufacturerId . ':' . urlencode($manufacturerAlias);

		if ($this->params->get('item_id')) {
			$link .= '&Itemid=' . (int)$this->params->get('item_id');
		}

		return $link;
	}

	/**
	 * @param   int     $tagId
	 * @param   string  $tagAlias
	 *
	 * @return string
	 *
	 * @since 4.0.3
	 */
	public function getTagRoute(int $tagId, string $tagAlias): string
	{
		$link = 'index.php?option=com_phocacart&view=items';
		$link .= '&tag=' . $tagId . ':' . urlencode($tagAlias);

		if ($this->params->get('item_id')) {
			$link .= '&Itemid=' . (int)$this->params->get('item_id');
		}

		return $link;
	}

	/**
	 * @inheritdoc
	 *
	 * @since 5.0.1
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onSchemaBeforeCompileHead' => ['onSchemaBeforeCompileHead', Priority::BELOW_NORMAL],
		];
	}

	/**
	 * Ads search markup to Schema.org
	 *
	 * @param   BeforeCompileHeadEvent  $event  The given event
	 *
	 * @return  void
	 *
	 * @since   5.0.1
	 */
	public function onSchemaBeforeCompileHead(BeforeCompileHeadEvent $event): void
	{
		if (!$this->params->get('schemaorg', 1)) {
			return;
		}

		$schema = $event->getSchema();
		$graph = $schema->get('@graph');

		foreach ($graph as &$entry) {
			if (!isset($entry['@type']) || $entry['@type'] !== 'WebSite') {
				continue;
			}

			$entry['potentialAction'] = [
				'@type'       => 'SearchAction',
				'target'      => Route::_('index.php?option=com_phocacart&view=items&search={search_term_string}', true, Route::TLS_IGNORE, true),
				'query-input' => 'required name=search_term_string',
			];
		}

		$schema->set('@graph', $graph);
	}
}
