<?php
/**
 * @package     PhocaCart
 * @subpackage  PhocaCart Console Plugin
 *
 * @copyright   (C) 2022 - Pavel Poles - n3t.cz
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Joomla\Plugin\Console\PhocaCart\Command\Products;

defined('_JEXEC') or die;

use Joomla\Plugin\Console\PhocaCart\Command\RecreateThumbnailsCommand;
use Joomla\Utilities\ArrayHelper;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class RecreateThumbnails extends RecreateThumbnailsCommand
{
	/**
	 * @inheritdoc
	 * @since 4.0.0
	 */
	protected static $defaultName = 'phocacart:products:recreatethumbnails';

	/**
	 * @inheritdoc
	 * @since 4.0.0
	 */
	protected $description = 'Recreate products thumbnails';

	/**
	 * @inheritdoc
	 * @since 4.0.0
	 */
	protected $help = "Recreates thumbnails for products.
			\nCould be limited to specific products or categories.";

	/**
	 * @inheritdoc
	 * @since 4.0.0
	 */
	protected $title = 'Products - Recreate thumbnails';

	/**
	 * Initialise the command.
	 *
	 * @return  void
	 *
	 * @since   4.0.0
	 */
	protected function configure(): void
	{
		parent::configure();

		$this->addOption('id', null, InputOption::VALUE_OPTIONAL, 'comma sepeareted list of product ids');
		$this->addOption('cat', null, InputOption::VALUE_OPTIONAL, 'comma separated list of category ids');
	}

	/**
	 * Internal function to execute the command.
	 *
	 * @param   InputInterface   $input   The input to inject into the command.
	 * @param   OutputInterface  $output  The output to inject into the command.
	 *
	 * @return  integer  The command exit code
	 *
	 * @since   4.0.0
	 */
	protected function doExecute(InputInterface $input, OutputInterface $output): int
	{
		$this->loadComponent();
		$this->configureIO($input, $output);
		$db = $this->getDatabase();
		$warnings = [];
		$successCount = 0;
		$itemsCount = 0;

		$query = $db->getQuery(true)
			->select($db->quoteName('p.id'))
			->select($db->quoteName('p.image'))
			->from($db->quoteName('#__phocacart_products') . ' as ' . $db->quoteName('p'))
			->where($db->quoteName('p.image') . ' <> ' . $db->quote(''));

		if ($ids = $input->getOption('id')) {
			$ids = explode(',', $ids);
			$ids = ArrayHelper::toInteger($ids);
			if ($ids)
				$query->where($db->quoteName('p.id') . ' in (' . implode(', ', $ids) . ')');
		}

		if ($categories = $input->getOption('cat')) {
			$categories = explode(',', $categories);
			$categories = ArrayHelper::toInteger($categories);
			if ($categories) {
				$query->join('INNER', $db->quoteName('#__phocacart_product_categories') . ' as ' . $db->quoteName('c'), $db->quoteName('c.product_id') . ' = ' . $db->quoteName('p.id'));
				$query->where($db->quoteName('c.category_id') . ' in (' . implode(', ', $categories) . ')');
			}
		}

		$db->setQuery($query);
		$items = $db->loadObjectList();
		$itemsCount += count($items);
		if ($items) {
			$this->ioStyle->info('Recreating main image thumbnails');
			$this->recreateThumbnails($items, 'productimage', $warnings, $successCount);
		}

		$query = $db->getQuery(true)
			->select($db->quoteName('p.product_id') . ' as ' . $db->quoteName('id'))
			->select($db->quoteName('p.image'))
			->from($db->quoteName('#__phocacart_product_images') . ' as ' . $db->quoteName('p'));

		if ($ids)
			$query->where($db->quoteName('p.product_id') . ' in (' . implode(', ', $ids) . ')');

		if ($categories) {
			$query->join('INNER', $db->quoteName('#__phocacart_product_categories') . ' as ' . $db->quoteName('c'), $db->quoteName('c.product_id') . ' = ' . $db->quoteName('p.product_id'));
			$query->where($db->quoteName('c.category_id') . ' in (' . implode(', ', $categories) . ')');
		}

		$db->setQuery($query);
		$items = $db->loadObjectList();
		$itemsCount += count($items);
		if ($items) {
			$this->ioStyle->info('Recreating additional image thumbnails');
			$this->recreateThumbnails($items, 'productimage', $warnings, $successCount);
		}

		if ($itemsCount == 0) {
			$this->ioStyle->info('Nothing to update');
			return self::RETURN_CODE_SUCCESSFUL;
		}

		if ($warnings) {
			$this->ioStyle->warning('Following errors occured');
			$this->ioStyle->listing($warnings);
		}

		if ($successCount) {
			$this->ioStyle->success($successCount . ' thumbnails recreated successfully');

			if ($warnings)
				return self::RETURN_CODE_PARTIALLY_SUCCESSFUL;
			else
				return self::RETURN_CODE_SUCCESSFUL;
		}

		return self::RETURN_CODE_FAILED;
	}
}