<?php
/**
 * @package     PhocaCart
 * @subpackage  PhocaCart Console Plugin
 *
 * @copyright   (C) 2022 - Pavel Poles - n3t.cz
 * @license     GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Joomla\Plugin\Console\PhocaCart\Command;

defined('_JEXEC') or die;

use Joomla\Console\Command\AbstractCommand;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;
use Joomla\Plugin\Console\PhocaCart\Command\BaseCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

abstract class ProductsCountCommand extends BaseCommand
{

	/**
	 * Type used for PhocacartCount::setProductCount function
	 * @var string
	 * @since 4.0.0
	 */
	protected $type = '';

	/**
	 * Loads list of items to be updated
	 *
	 * @return array
	 *
	 * @since 4.0.0
	 */
	abstract protected function loadItems(): array;

	/**
	 * Internal function to execute the command.
	 *
	 * @param   InputInterface   $input   The input to inject into the command.
	 * @param   OutputInterface  $output  The output to inject into the command.
	 *
	 * @return  integer  The command exit code
	 *
	 * @since   4.0.0
	 */
	protected function doExecute(InputInterface $input, OutputInterface $output): int
	{
		$this->loadComponent();
		$this->configureIO($input, $output);

		$items = $this->loadItems();

		if (!$items) {
			$this->ioStyle->info('Nothing to update');
			return self::RETURN_CODE_SUCCESSFUL;
		}

		$errorList = [];
		$successCount = 0;
		$this->ioStyle->progressStart(count($items));
		foreach ($items as $item) {
			$this->ioStyle->debugLn($item->id . ' - ' . $item->title);

			if (!\PhocacartCount::setProductCount([$item->id], $this->type, 1))
				$errorList[] = $item;
			else
				$successCount++;

			$this->ioStyle->progressAdvance();
		}
		$this->ioStyle->progressFinish();

		if ($errorList) {
			array_walk($errorList, function(&$category) {
				$category = $category->id . ' - ' . $category->title;
			});
			$this->ioStyle->warning('Following items were not updated');
			$this->ioStyle->listing($errorList);
		}

		if ($successCount) {
			$this->ioStyle->success('Products count recalculated in ' . $successCount . ' items');

			if ($errorList)
				return self::RETURN_CODE_PARTIALLY_SUCCESSFUL;
			else
				return self::RETURN_CODE_SUCCESSFUL;
		}

		return self::RETURN_CODE_FAILED;
	}
}