<?php
/**
 * @package    Install from Bongovo! plugin
 *
 * @author     Pavel Poles - n3t.cz
 * @copyright  © 2022 - 2026 Pavel Poles - n3t.cz. All rights reserved.
 * @license    GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 * @link       https://n3t.bitbucket.io
 **/

defined('_JEXEC') or die;

use Joomla\CMS\Access\Exception\NotAllowed;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Version;
use Joomla\CMS\Extension\ExtensionHelper;
use Joomla\Registry\Registry;
use Joomla\CMS\Language\LanguageHelper;
use Joomla\CMS\Filter\InputFilter;


class plgInstallerBongovo extends CMSPlugin
{
	private const URL = 'https://www.bongovo.cz/download/translationlist.xml';
	private const USER_AGENT = 'Bongovo.cz Joomla Installer plugin';

	/**
	 * The loaded extensions.
	 *
	 * @var    array
	 * @since  4.0.0
	 */
	private static $loadedExtensions = [];

	public static function getExtensionRecord(string $element, string $type, ?int $clientId = null, ?string $folder = null): ?\stdClass
	{
		if (Version::MAJOR_VERSION > 3)
			return ExtensionHelper::getExtensionRecord($element, $type, $clientId, $folder);

		if ($type === 'plugin' && $folder === null)
		{
			throw new \InvalidArgumentException(sprintf('`$folder` is required when `$type` is `plugin` in %s()', __METHOD__));
		}

		if (\in_array($type, ['module', 'language', 'template'], true) && $clientId === null)
		{
			throw new \InvalidArgumentException(
				sprintf('`$clientId` is required when `$type` is `module`, `language` or `template` in %s()', __METHOD__)
			);
		}

		$key = $element . '.' . $type . '.' . $clientId . '.' . $folder;

		if (!\array_key_exists($key, self::$loadedExtensions))
		{
			$db    = Factory::getDbo();
			$query = $db->getQuery(true)
				->select('*')
				->from($db->quoteName('#__extensions'))
				->where(
					[
						$db->quoteName('element') . ' = ' . $db->quote($element),
						$db->quoteName('type') . ' = ' . $db->quote($type),
					]
				);

			if ($clientId !== null)
				$query->where($db->quoteName('client_id') . ' = ' . $clientId);

			if ($folder !== null)
				$query->where($db->quoteName('folder') . ' = ' . $db->quote($folder));

			$query->setLimit(1);
			$db->setQuery($query);

			self::$loadedExtensions[$key] = $db->loadObject();
		}

		return self::$loadedExtensions[$key];
	}

	public function loadTranslations(): array
	{
		static $translations = null;

		if ($translations === null)
		{
			$translations = [];

			$http = HttpFactory::getHttp();
			$http->setOption('User-Agent', self::USER_AGENT);
			$response = $http->get(self::URL);

			if ($response->code !== 200)
				return [];

			if (!$response->body)
				return [];

			try
			{
				$xml = new \SimpleXMLElement($response->body);
			}
			catch (\Exception $exception)
			{
				return [];
			}

			$version   = (new Version)->getShortVersion();
			$languages = LanguageHelper::getInstalledLanguages();
			$languages = array_keys($languages[0]) + array_keys($languages[1]);

			foreach ($xml->xpath('extension') as $extension) {
				$translation = current((array)$extension->attributes());
				$translation = array_merge([
					'name'                => null,
					'extensiontype'       => null,
					'extensionfolder'     => null,
					'extensionclient'     => null,
					'extensionelement'    => null,
					'extensionminversion' => null,
					'extensionmaxversion' => null,
					'language'            => null,
					'element'             => null,
					'type'                => 'file',
					'version'             => null,
					'description'         => null,
					'downloadurl'         => null,
					'joomlaminversion'    => null,
					'joomlamaxversion'    => null,

					'extensionversion'        => null,
					'extensionversionwarning' => null,
					'installedversion'        => null,
					'needupdate'              => false,
					'languageinstalled'       => false,
				], $translation);

				$translation = (object) $translation;

				if ($translation->joomlaminversion)
				{
					if (!version_compare($version, $translation->joomlaminversion, '>='))
						continue;
				}

				if ($translation->joomlamaxversion)
				{
					if (!version_compare($version, $translation->joomlamaxversion, '<='))
						continue;
				}

				if ($translation->element && $translation->type)
				{
					$extension = $this->getExtensionRecord($translation->element, $translation->type);
					if ($extension)
					{
						$manifest                      = new Registry($extension->manifest_cache);
						$translation->installedversion = $manifest->get('version');
						$translation->needupdate       = $translation->version && version_compare($translation->installedversion, $translation->version, '<');
					}
				}


				if ($translation->extensionelement && $translation->extensiontype)
				{
					$clientId = null;
					$folder = null;
					if (in_array($translation->extensiontype, ['module', 'language', 'template']))
						$clientId = $translation->extensionclient == 'site' ? 0 : 1;
					if (in_array($translation->extensiontype, ['plugin']))
						$folder = $translation->extensionfolder;
					$extension = $this->getExtensionRecord($translation->extensionelement, $translation->extensiontype, $clientId, $folder);
					if ($extension)
					{
						$manifest                      = new Registry($extension->manifest_cache);
						$translation->extensionversion = $manifest->get('version');

						if (
							$translation->extensionminversion && version_compare($translation->extensionversion, $translation->extensionminversion, '<')
							&& $translation->extensionmaxversion && version_compare($translation->extensionversion, $translation->extensionmaxversion, '>')
						)
							$translation->extensionversionwarning = Text::sprintf('PLG_INSTALLER_BONGOVO_EXTENSION_MINMAX_VERSION', $translation->extensionminversion, $translation->extensionmaxversion, $translation->extensionversion);
						elseif ($translation->extensionminversion && version_compare($translation->extensionversion, $translation->extensionminversion, '<'))
							$translation->extensionversionwarning = Text::sprintf('PLG_INSTALLER_BONGOVO_EXTENSION_MIN_VERSION', $translation->extensionminversion, $translation->extensionversion);
						elseif ($translation->extensionmaxversion && version_compare($translation->extensionversion, $translation->extensionmaxversion, '>'))
							$translation->extensionversionwarning = Text::sprintf('PLG_INSTALLER_BONGOVO_EXTENSION_MAX_VERSION', $translation->extensionmaxversion, $translation->extensionversion);
					}
				}

				if ($translation->language)
				{
					$translation->languageinstalled = in_array($translation->language, $languages);
				}

				$translations[] = $translation;
			}

			usort($translations, function($a, $b) {
				return strcasecmp($a->name, $b->name);
			});
		}

		return $translations;
	}

	public function loadTranslationsToInstall(): array
	{
		$translations = $this->loadTranslations();

		return array_filter($translations, function ($translation) {
			return $translation->needupdate ||
				($translation->extensionversion
				&& $translation->languageinstalled
				&& !$translation->installedversion);
		});
	}

	public function loadTranslationsInstalled(): array
	{
		$translations = $this->loadTranslations();

		return array_filter($translations, function ($translation) {
			return !!$translation->installedversion && !$translation->needupdate;
		});
	}

	public function loadTranslationsOther(): array
	{
		$translations = $this->loadTranslations();

		return array_filter($translations, function ($translation) {
			return (!$translation->extensionversion
					|| !$translation->languageinstalled)
				&& !$translation->installedversion;
		});
	}

	public function onInstallerAddInstallationTab()
	{
		$this->loadLanguage();

		$tab          = array();
		$tab['name']  = 'bongovo';
		$tab['label'] = Text::_('PLG_INSTALLER_BONGOVO_TEXT');

		ob_start();
		include PluginHelper::getLayoutPath('installer', 'bongovo');
		$tab['content'] = ob_get_clean();

		return $tab;
	}

	public function onAjaxBongovo()
	{
		$this->loadLanguage();
		$app = Factory::getApplication();
		if (!$app->isClient('administrator'))
			throw new NotAllowed(Text::_('PLG_INSTALLER_BONGOVO_EXCEPTION_UNAUTHORIED'));

		if (Version::MAJOR_VERSION >= 4)
			$user = $app->getIdentity();
		else
			$user = Factory::getUser();
		if ($user->guest)
			throw new NotAllowed();

		if (Version::MAJOR_VERSION >= 4)
			$input = $app->getInput();
		else
			$input = $app->input;

		$url = $input->get('url', null, 'url');
		if (!$url)
			throw new \InvalidArgumentException(Text::_('PLG_INSTALLER_BONGOVO_EXCEPTION_INPUT'));

		$http = HttpFactory::getHttp();
		$http->setOption('User-Agent', self::USER_AGENT);
		$response = $http->get($url);

		if ($response->code !== 200)
			throw new \Exception(Text::_('PLG_INSTALLER_BONGOVO_EXCEPTION_HTTP'));

		if (!$response->body)
			throw new \Exception(Text::_('PLG_INSTALLER_BONGOVO_EXCEPTION_HTTP'));

		try
		{
			$xml = new \SimpleXMLElement($response->body);
		}
		catch (\Exception $exception)
		{
			return null;
		}

		$version = (new Version)->getShortVersion();
		$product = strtolower(InputFilter::getInstance()->clean(Version::PRODUCT, 'cmd'));

		$updates = [];
		foreach ($xml->xpath('update') as $update)
		{
			if (!count($update->xpath('downloads/downloadurl[@type="full"]')))
				continue;

			if (count($update->xpath('targetplatform')))
			{
				$values = current($update->xpath('targetplatform')[0]->attributes());
				if (!array_key_exists('name', $values))
					$values['name'] = $product;
				if (!array_key_exists('version', $values))
					$values['name'] = $version;

				if ($product != $values['name'] || !preg_match('/^' . $values['version'] . '/', $version))
				{
					continue;
				}
			}

			$updates[] = [
				'version' => $update->version,
				'url'     => $update->downloads->downloadurl,
			];
		}

		if (count($updates))
		{
			usort($updates, function ($a, $b) {
				if ($a['version'] == $b['version'])
					return 0;

				return version_compare($a['version'], $b['version'], '<') ? -1 : 1;
			});

			return (string) $updates[0]['url'];
		}

		throw new \Exception(Text::_('PLG_INSTALLER_BONGOVO_EXCEPTION_DOWNLOAD_URL'));
	}

}
